// ==UserScript==
// @name         Google Flow API v4 (WebSocket)
// @namespace    http://tampermonkey.net/
// @version      4.0
// @description  Google Flow API - WebSocket 서버와 연결하여 외부 API로 사용
// @author       You
// @match        https://labs.google/fx/tools/flow*
// @grant        none
// ==/UserScript==

(function() {
    'use strict';

    var WS_URL = 'ws://localhost:8002/ws';
    var RECONNECT_INTERVAL = 5000;

    window.FlowAPI = {
        pendingConfig: null,
        responseCallback: null,
        _installed: false,
        _ws: null,
        _wsConnected: false,

        // WebSocket 연결
        connectWebSocket: function() {
            var self = this;

            if (this._ws && this._ws.readyState === WebSocket.OPEN) {
                return;
            }

            console.log('[FlowAPI] Connecting to WebSocket server...');

            try {
                this._ws = new WebSocket(WS_URL);

                this._ws.onopen = function() {
                    self._wsConnected = true;
                    console.log('%c[FlowAPI] WebSocket connected to server!', 'color: #4CAF50; font-weight: bold');
                };

                this._ws.onclose = function() {
                    self._wsConnected = false;
                    console.log('[FlowAPI] WebSocket disconnected. Reconnecting in 5s...');
                    setTimeout(function() { self.connectWebSocket(); }, RECONNECT_INTERVAL);
                };

                this._ws.onerror = function(err) {
                    console.error('[FlowAPI] WebSocket error:', err);
                };

                this._ws.onmessage = function(event) {
                    try {
                        var data = JSON.parse(event.data);
                        self.handleServerMessage(data);
                    } catch (e) {
                        console.error('[FlowAPI] Message parse error:', e);
                    }
                };

            } catch (e) {
                console.error('[FlowAPI] WebSocket connection failed:', e);
                setTimeout(function() { self.connectWebSocket(); }, RECONNECT_INTERVAL);
            }
        },

        // 서버 메시지 처리
        handleServerMessage: async function(data) {
            var self = this;

            if (data.action === 'generate') {
                console.log('[FlowAPI] Received generate request from server');

                try {
                    var result = await this.generate(data.params);
                    this.sendToServer({
                        requestId: data.requestId,
                        result: result
                    });
                } catch (e) {
                    this.sendToServer({
                        requestId: data.requestId,
                        result: { success: false, error: e.message }
                    });
                }
            }

            if (data.type === 'pong') {
                console.log('[FlowAPI] Pong received');
            }
        },

        // 서버로 메시지 전송
        sendToServer: function(data) {
            if (this._ws && this._ws.readyState === WebSocket.OPEN) {
                this._ws.send(JSON.stringify(data));
            }
        },

        // fetch 인터셉터 설치
        install: function() {
            if (this._installed) return;
            this._installed = true;

            var self = this;
            var origFetch = window.fetch;

            window.fetch = async function(url, opts) {
                var urlStr = typeof url === 'string' ? url : url.toString();

                if (urlStr.indexOf('batchGenerateImages') !== -1 && opts && opts.body) {
                    if (self.pendingConfig) {
                        var body = JSON.parse(opts.body);
                        var cfg = self.pendingConfig;
                        self.pendingConfig = null;

                        console.log('[FlowAPI] Modifying request:', cfg);

                        var targetCount = cfg.count || body.requests.length;
                        while (body.requests.length < targetCount) {
                            var clone = JSON.parse(JSON.stringify(body.requests[0]));
                            clone.seed = Math.floor(Math.random() * 1000000);
                            body.requests.push(clone);
                        }
                        body.requests = body.requests.slice(0, targetCount);

                        body.requests.forEach(function(req) {
                            if (cfg.prompt) req.prompt = cfg.prompt;
                            if (cfg.model) req.imageModelName = cfg.model;
                            if (cfg.aspectRatio) req.imageAspectRatio = cfg.aspectRatio;
                        });

                        opts.body = JSON.stringify(body);
                        console.log('[FlowAPI] Request modified:', body.requests.length, 'images');
                    }

                    var resp = await origFetch(url, opts);
                    var clone = resp.clone();

                    clone.json().then(function(data) {
                        if (self.responseCallback) {
                            self.responseCallback(data);
                            self.responseCallback = null;
                        }
                    }).catch(function(e) {
                        console.error('[FlowAPI] Response parse error:', e);
                    });

                    return resp;
                }

                return origFetch(url, opts);
            };

            console.log('[FlowAPI] Interceptor installed');
        },

        _triggerGenerate: function() {
            var btns = document.querySelectorAll('button');
            for (var i = 0; i < btns.length; i++) {
                if (btns[i].textContent.indexOf('arrow_forward') !== -1) {
                    btns[i].click();
                    return true;
                }
            }
            return false;
        },

        _setTextarea: function(text) {
            var ta = document.querySelector('textarea');
            if (!ta) return false;
            var setter = Object.getOwnPropertyDescriptor(HTMLTextAreaElement.prototype, 'value').set;
            setter.call(ta, text);
            ta.dispatchEvent(new Event('input', { bubbles: true }));
            return true;
        },

        // aspectRatio 정규화
        _normalizeAspectRatio: function(ratio) {
            if (!ratio) return 'IMAGE_ASPECT_RATIO_LANDSCAPE';
            ratio = ratio.toLowerCase();
            if (ratio === 'square' || ratio === '1:1' || ratio === '1_1') return 'IMAGE_ASPECT_RATIO_SQUARE';
            if (ratio === 'landscape' || ratio === '16:9' || ratio === '16_9') return 'IMAGE_ASPECT_RATIO_LANDSCAPE';
            if (ratio === 'portrait' || ratio === '9:16' || ratio === '9_16') return 'IMAGE_ASPECT_RATIO_PORTRAIT';
            return ratio;
        },

        generate: function(options) {
            var self = this;
            this.install();

            var prompt = typeof options === 'string' ? options : options.prompt;
            var count = (options && options.count) || 2;
            var model = (options && options.model) || 'GEM_PIX_2';
            var aspectRatio = this._normalizeAspectRatio(options && options.aspectRatio);

            return new Promise(function(resolve, reject) {
                self.pendingConfig = {
                    prompt: prompt,
                    count: count,
                    model: model,
                    aspectRatio: aspectRatio
                };

                self.responseCallback = function(data) {
                    if (data.error) {
                        resolve({ success: false, error: data.error });
                        return;
                    }

                    var imgs = (data.media || []).map(function(m) {
                        var g = m.image && m.image.generatedImage;
                        return g ? { url: g.fifeUrl, prompt: g.prompt, seed: g.seed } : null;
                    }).filter(Boolean);

                    resolve({ success: true, count: imgs.length, images: imgs });
                };

                if (!self._setTextarea('generating...')) {
                    self.pendingConfig = null;
                    self.responseCallback = null;
                    reject(new Error('Textarea not found'));
                    return;
                }

                setTimeout(function() {
                    if (!self._triggerGenerate()) {
                        self.pendingConfig = null;
                        self.responseCallback = null;
                        reject(new Error('Generate button not found'));
                    }
                }, 100);

                setTimeout(function() {
                    if (self.responseCallback) {
                        self.responseCallback = null;
                        reject(new Error('Timeout 60s'));
                    }
                }, 60000);
            });
        }
    };

    // 초기화
    FlowAPI.install();
    FlowAPI.connectWebSocket();

    // 핑 주기적 전송 (연결 유지)
    setInterval(function() {
        if (FlowAPI._wsConnected) {
            FlowAPI.sendToServer({ type: 'ping' });
        }
    }, 30000);

    // 콘솔 로그
    console.log('%c' + [
        '',
        ' ╔═══════════════════════════════════════════════════════════╗',
        ' ║   FlowAPI v4.0 (WebSocket) - Tampermonkey 주입 성공!      ║',
        ' ╚═══════════════════════════════════════════════════════════╝',
        ''
    ].join('\n'), 'color: #4CAF50; font-weight: bold');

    console.log('%cWebSocket 서버 연결 중... (ws://localhost:8002/ws)', 'color: #2196F3');
    console.log('%c서버 실행: python flow_api_server.py', 'color: #FFC107');
    console.log('%cAPI 문서: http://localhost:8002/docs', 'color: #FFC107');

})();
